#### Preparation ####

# Emptying the environment to prevent confusion and errors due to leftovers from previous analyses:

rm(list = ls())

# Checking whether the working directory is correct:

getwd()

# If working directory is not correct, reset the working directory:

setwd()

### Please note that the working directory should have a folder named "data" with the csv.files in
### it and a folder named "output" (into which results will be saved) in order for the script to work.

# Loading the required packages:

install.packages("QCA")
install.packages("SetMethods")
install.packages("readxl")

library(QCA)
library(SetMethods)
library(readxl)

# Loading the required data for the analysis:

ISOLATEDLIFTRAW <- read.csv("data/ISOLATEDLIFTRAW.csv", row.names = 1)

ISOLATEDLIFTRAW

### Calibration ####

# Condition "LOWALTOPS":

LOWALTOPS <- calibrate(ISOLATEDLIFTRAW$A_Einstiegshoehe, 
                       type = "fuzzy", 
                       thresholds = c(1600, 1300.1, 1000), 
                       logistic = TRUE)

plot(ISOLATEDLIFTRAW$A_Einstiegshoehe, 
     LOWALTOPS, 
     main = "Calibration of LOWALTOPS",
     xlab = "Raw score",
     ylab = "Calibrated score")
abline(h = 0.5, 
       v = 1300)

skew.check(LOWALTOPS, hist = TRUE)

LOWALTOPS== 0.5

### not skewed

# Condition "LOWELEVDIFF":

LOWELEVDIFF <- calibrate(ISOLATEDLIFTRAW$B_HoeheDiff, 
                         type = "fuzzy", 
                         thresholds = c(500, 200, 75), 
                         logistic = TRUE)

plot(ISOLATEDLIFTRAW$B_HoeheDiff, 
     LOWELEVDIFF, 
     main = "LOWELEVDIFF",
     xlab = "Raw score",
     ylab = "Calibrated score")
abline(h = 0.5, 
       v = 200)

skew.check(LOWELEVDIFF, hist = TRUE)

LOWELEVDIFF == 0.5

### not skewed

# Condition "LOWSNOWFEB":

LOWSNOWFEB <- calibrate(ISOLATEDLIFTRAW$C_Schneemenge_FEB,
                        type = "fuzzy",
                        thresholds = c(70, 49.9, 30),
                        logistic = TRUE)

plot(ISOLATEDLIFTRAW$C_Schneemenge_FEB, 
     LOWSNOWFEB, 
     main = "Calibration of LOWSNOWFEB",
     xlab = "Raw score",
     ylab = "Calibrated score")
abline(h = 0.5, 
       v = 50)

skew.check(LOWSNOWFEB, hist = TRUE)

## not skewed

# Condition "NOREPLACEMENT":

REPLACEMENT <- ISOLATEDLIFTRAW$D_REPLACEMENT

NOREPLACEMENT <- 1-REPLACEMENT

skew.check(NOREPLACEMENT, hist = TRUE)

## skewed, will lead to low RoN

# Condition "NOSNOWMAKING":

SNOWMAKING <- ISOLATEDLIFTRAW$E_SNOW_MAKING

NOSNOWMAKING <- 1-SNOWMAKING

skew.check(NOSNOWMAKING, hist = TRUE)

## skewed, will lead to low RoN

# Condition "HIGHCOMPETITION":

HIGHCOMPETITION <- ISOLATEDLIFTRAW$F_EXCL_COMP_3_0

skew.check(HIGHCOMPETITION, hist = TRUE)

## not skewed

# Outcome "LSAP":

LSAP <- ISOLATEDLIFTRAW$Y_LSAP

skew.check(LSAP, hist = TRUE)

## not skewed

# Combining the calibrated data into a new dataframe:

ISOLATEDLIFTCAL <- data.frame(LOWALTOPS, LOWELEVDIFF, LOWSNOWFEB, NOREPLACEMENT, NOSNOWMAKING, HIGHCOMPETITION, LSAP)

row.names(ISOLATEDLIFTCAL) <- ISOLATEDLIFTRAW$SKI_AREA

head(ISOLATEDLIFTCAL)

write.csv(ISOLATEDLIFTCAL, file = "data/ISOLATEDLIFTCAL.csv", row.names = TRUE)

#### Standard Analysis ####

### Analysis of Necessity ####

## LSAP Outcome ####

# Obtaining the parameters of fit for single necessary conditions for the LSAP outcome:

QCAfit(x = ISOLATEDLIFTCAL[,c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION")], 
       y = ISOLATEDLIFTCAL$LSAP)

## Non-LSAP Outcome ####

# Obtaining the parameters of fit for single necessary conditions for the non-occurrence of the outcome:

QCAfit(x = ISOLATEDLIFTCAL[,c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION")], 
       y = 1-ISOLATEDLIFTCAL$LSAP)

## SUIN-conditions for both outcomes

# SUIN conditions for the occurrence of the outcome:

SS_Y <- superSubset(data = ISOLATEDLIFTCAL, 
                    outcome = "LSAP", 
                    conditions = c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION"), 
                    ron.cut = 0.5,
                    cov.cut = 0.6,
                    incl.cut = 0.9,
                    depth = 2)

# No SUIN conditions.

# SUIN conditions for the non-occurrence of the outcome:

SS_NY <- superSubset(data = ISOLATEDLIFTCAL, 
                     outcome = "~LSAP", 
                     conditions = c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION"), 
                     ron.cut = 0.5,
                     cov.cut = 0.6,
                     incl.cut = 0.9,
                     depth = 2)

# No SUIN conditions.

### Analysis of Sufficiency ####

## LSAP outcome ####

## Truth table:

TTfs_y <- truthTable(ISOLATEDLIFTCAL, 
                     outcome = "LSAP",
                     conditions = c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION"),
                     incl.cut = 0.745,
                     pri.cut = 0.5,
                     sort.by = c("OUT", "incl"), 
                     show.cases=TRUE,
                     complete = FALSE)
TTfs_y

# Copying this truth table into a text editor:

stargazerTT(TTfs_y, show.cases = TRUE, type = "text", out = "output/TTfs_yisolatedlift.txt")

# Creating the conservative (complex) solution:

sol_yc <- minimize(TTfs_y, details=TRUE, show.cases=TRUE, use.tilde=TRUE)
sol_yc

# Saving the solution in a .txt-file

stargazerSol(sol_yc, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_ycisolatedlift.txt")

# Creating the parsimonious solution for the LSAP outcome:

sol_yp <- minimize(TTfs_y, 
                   details = TRUE, 
                   include = "?",
                   use.tilde = TRUE,
                   row.dom = TRUE)
sol_yp

# Saving the solution in a .txt-file

stargazerSol(sol_yp, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_ypisolatedlift.txt")

# Creating the intermediate solution for the LSAP outcome: 

sol_yi <- minimize(TTfs_y, 
                   details = TRUE, 
                   include = "?",
                   row.dom = TRUE,
                   dir.exp = c("NOSNOWMAKING", "HIGHCOMPETITION", "LOWELEVDIFF", "LOWALTOPS", "NOREPLACEMENT", "LOWSNOWFEB"),
                   use.tilde = TRUE)

sol_yi

# Saving the solution in a .txt-file

stargazerSol(sol_yi, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_yiisolatedlift.txt")

## Non-LSAP outcome ####

# Truth Table:

TTfs_ny <- truthTable(ISOLATEDLIFTCAL, 
                      outcome = "~LSAP",
                      conditions = c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION"),
                      incl.cut = 0.75,
                      pri.cut = 0.5,
                      sort.by = c("OUT", "incl"), 
                      show.cases=TRUE,
                      complete = FALSE)
TTfs_ny

# Copying this truth table into a text editor:

stargazerTT(TTfs_ny, show.cases = TRUE, type = "text", out = "output/TTfs_nyisolatedlift.txt")

# Creating the conservative solution:

sol_nyc <- minimize(TTfs_ny, details=TRUE, show.cases=TRUE, use.tilde=TRUE, row.dom = TRUE) 

sol_nyc

# Saving the solution in a .txt-file

stargazerSol(sol_nyc, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nycisolatedlift.txt")

# Creating the parsimonious solution:

sol_nyp <- minimize(TTfs_ny, 
                    details = TRUE, 
                    include = "?",
                    use.tilde = TRUE,
                    row.dom = TRUE)
sol_nyp

# Saving the solution in a .txt-file

stargazerSol(sol_nyp, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nypisolatedlift.txt")

# Creating the intermediate solution:

sol_nyi <- minimize(TTfs_ny, 
                    details = TRUE, 
                    include = "?",
                    row.dom = TRUE,
                    dir.exp = c("~NOSNOWMAKING", "~HIGHCOMPETITION", "~LOWELEVDIFF", "~LOWALTOPS", "~NOREPLACEMENT", "~LOWSNOWFEB"),
                    use.tilde = TRUE)
sol_nyi

# Saving the solution in a .txt-file

stargazerSol(sol_nyi, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nyisolatedlift.txt")

### Enhanced Standard Analysis ####

## Contradictory Assumptions ####

# Checking whether there are contradictory assumptions for the conservative solutions
# for the occurrence and the non-occurrence of the outcome.

CSAc <- intersect(rownames(sol_nyc$SA$M1), rownames(sol_yc$SA$M1))
CSAc

# Checking whether there are contradictory assumptions for the parsimonious solutions
# for the occurrence and the non-occurrence of the outcome.

CSAp <- intersect(rownames(sol_nyp$SA$M1), rownames(sol_yp$SA$M1))
CSAp

# Checking whether there are contradictory assumptions for the intermediate solutions
# for the occurrence and the non-occurrence of the outcome.

CSAi <- intersect(rownames(sol_nyi$SA$M1), rownames(sol_yi$SA$M1))
CSAi

## Simultaneous Subset Relations ####

# Checking for simultaneous subset relations in the truth tables.

SSR<-intersect(rownames(TTfs_ny$tt)[TTfs_ny$tt$OUT==1],rownames(TTfs_y$tt)[TTfs_y$tt$OUT==1])
SSR

# There are no simultaneous subset relations.

# Enhanced Standard Analysis ####

## LSAP Outcome ####

TTfs_yesa <- esa(oldtt = TTfs_y,
                 contrad_rows = c(CSAc, CSAp, CSAi))

TTfs_yesa

# Copying this truth table into a text editor:

stargazerTT(TTfs_yesa, show.cases = TRUE, type = "text", out = "output/TTfs_yesaisolatedlift.txt")

# Creating the conservative (complex) solution:

sol_ycesa <- minimize(TTfs_yesa, details=TRUE, show.cases=TRUE, use.tilde=TRUE)
sol_ycesa

# Saving the solution in a .txt-file

stargazerSol(sol_ycesa, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_ycesaisolatedlift.txt")

# Creating the parsimonious solution:

sol_ypesa <- minimize(TTfs_yesa, 
                      details = TRUE, 
                      include = "?",
                      use.tilde = TRUE,
                      row.dom = TRUE)
sol_ypesa

# Saving the solution in a .txt-file

stargazerSol(sol_ypesa, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_ypesaisolatedlift.txt")

# Creating the intermediate solution: 

sol_yiesa <- minimize(TTfs_yesa, 
                      details = TRUE, 
                      include = "?",
                      row.dom = TRUE,
                      dir.exp = c("NOSNOWMAKING", "HIGHCOMPETITION", "LOWELEVDIFF", "LOWALTOPS", "NOREPLACEMENT", "LOWSNOWFEB"),
                      use.tilde = TRUE)

sol_yiesa

# Saving the solution in a .txt-file

stargazerSol(sol_yiesa, outcome = "LSAP", show.cases = TRUE, type = "text", out = "output/sol_yiesaisolatedlift.txt")

# Non-LSAP-OUTCOME ####

TTfs_nyesa <- esa(oldtt = TTfs_ny,
                  contrad_rows = c(CSAc, CSAp, CSAi))

TTfs_nyesa

# Copying this truth table into a text editor:

stargazerTT(TTfs_nyesa, show.cases = TRUE, type = "text", out = "output/TTfs_nyesaisolatedlift.txt")

# Creating the conservative solution:

sol_nycesa <- minimize(TTfs_nyesa, details=TRUE, show.cases=TRUE, use.tilde=TRUE, row.dom = TRUE) 

sol_nycesa

# Saving the solution in a .txt-file

stargazerSol(sol_nycesa, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nycesaisolatedlift.txt")

# Crating the parsimonious solution:

sol_nypesa <- minimize(TTfs_nyesa, 
                       details = TRUE, 
                       include = "?",
                       use.tilde = TRUE,
                       row.dom = TRUE)
sol_nypesa

# Saving the solution in a .txt-file

stargazerSol(sol_nypesa, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nypesaisolatedlift.txt")

# Creating the intermediate solution:

sol_nyiesa <- minimize(TTfs_nyesa, 
                       details = TRUE, 
                       include = "?",
                       row.dom = TRUE,
                       dir.exp = c("~NOSNOWMAKING", "~HIGHCOMPETITION", "~LOWELEVDIFF", "~LOWALTOPS", "~NOREPLACEMENT", "~LOWSNOWFEB"),
                       use.tilde = TRUE)
sol_nyiesa

# Saving the solution in a .txt-file

stargazerSol(sol_nyiesa, outcome = "~LSAP", show.cases = TRUE, type = "text", out = "output/sol_nyiesaisolatedlift.txt")

#### Robustness Test ####

### LSAP Outcome ####

# Saving the initial solution for further testing

IS_y <- sol_ypesa
IS_ny <- sol_nypesa

# Saving the conditions for further testing

conds <- c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION")

# Testing calibration

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "A_Einstiegshoehe",
  test.cond.calib = "LOWALTOPS",
  test.thresholds = c(1600, 1300.1, 1000),
  step = 50,
  max.runs = 40,
  outcome  = "LSAP",
  conditions =  c("LOWALTOPS", "LOWELEVDIFF", "LOWSNOWFEB", "NOREPLACEMENT", "NOSNOWMAKING", "HIGHCOMPETITION"),
  incl.cut = 0.745,
  n.cut = 1,
  include = "?"
)

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "B_HoeheDiff",
  test.cond.calib = "LOWELEVDIFF",
  test.thresholds = c(500, 200, 75), 
  step = 25,
  max.runs = 40,
  outcome  = "LSAP",
  conditions =  conds,
  incl.cut = 0.745,
  n.cut = 1,
  include = "?"
)

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "C_Schneemenge_FEB",
  test.cond.calib = "LOWSNOWFEB",
  test.thresholds = c(70, 49.9, 30),
  step = 5,
  max.runs = 40,
  outcome  = "LSAP",
  conditions =  conds,
  incl.cut = 0.745,
  n.cut = 1,
  include = "?"
)

# Testing consistency cut-off

rob.inclrange(
  data = ISOLATEDLIFTCAL,
  step = 0.01,
  max.runs = 40,
  outcome  = "LSAP",
  conditions =  conds,
  incl.cut = 0.745,
  n.cut = 1,
  include = "?"
)

# Testing n-cut

rob.ncutrange(
  data = ISOLATEDLIFTCAL,
  step = 1,
  max.runs = 20,
  outcome  = "LSAP",
  conditions =  conds,
  incl.cut = 0.745,
  n.cut = 1,
  include = "?"
)

# Creating test solutions

# Altering the consistency threshold

TSy1 <- minimize(data = ISOLATEDLIFTCAL,
                 outcome  = "LSAP",
                 conditions = conds,
                 incl.cut = 0.7,
                 n.cut = 1,
                 include = "?",
                 details = TRUE, show.cases = TRUE)

TSy1

ISOLATEDLIFTCAL2 <- ISOLATEDLIFTCAL

ISOLATEDLIFTCAL2$LOWALTOPS <- calibrate(ISOLATEDLIFTRAW$A_Einstiegshoehe, 
                                   type = "fuzzy", 
                                   thresholds = c(1900, 1500, 1100), 
                                   logistic = TRUE) 

ISOLATEDLIFTCAL2$LOWSNOWFEB<- calibrate(ISOLATEDLIFTRAW$C_Schneemenge_FEB,
                                   type = "fuzzy",
                                   thresholds = c(100, 70.1, 49.9),
                                   logistic = TRUE)

TSy2 <- minimize(data = ISOLATEDLIFTCAL2,
                 outcome  = "LSAP",
                 conditions = conds,
                 incl.cut = 0.745,
                 n.cut = 1,
                 include = "?",
                 details = TRUE, show.cases = TRUE)


TSy2

TSy3 <- minimize(data = ISOLATEDLIFTCAL2,
                 outcome  = "LSAP",
                 conditions = conds,
                 incl.cut = 0.745,
                 n.cut = 2,
                 include = "?",
                 details = TRUE, show.cases = TRUE)

TSy3

intersection(TSy1, TSy2, TSy3)

# Creating the test set in a list:
TSy <- list(TSy1, TSy2, TSy3)

# Obtaining Boolean expression for RC:
intersection(IS_y, TSy1, TSy2, TSy3)

# Calculating parameters for the robust core:

rob.corefit(test_sol = TSy, 
            initial_sol = IS_y, 
            outcome = "LSAP")

RFy <- rob.fit(test_sol = TSy, 
            initial_sol = IS_y, 
            outcome = "LSAP") 

RFy

# Plotting the initial solution against the test set:

rob.xyplot(test_sol = TSy, 
           initial_sol = IS_y, 
           outcome = "LSAP", 
           fontsize = 5, 
           all_labels = FALSE,
           jitter=TRUE,
           area_lab = TRUE)

# Looking at cases against the test set:

rob.cases(test_sol = TSy, 
          initial_sol = IS_y, 
          outcome = "LSAP")


# Worse test solution:

rob.singletest(test_sol = TSy, initial_sol = IS_y, outcome = "LSAP")

# Non-LSAP-Outcome ####

# Testing calibration

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "A_Einstiegshoehe",
  test.cond.calib = "LOWALTOPS",
  test.thresholds = c(1600, 1300.1, 1000),
  step = 50,
  max.runs = 40,
  outcome  = "~LSAP",
  conditions =  conds,
  incl.cut = 0.75,
  n.cut = 1,
  include = "?"
)

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "B_HoeheDiff",
  test.cond.calib = "LOWELEVDIFF",
  test.thresholds = c(500, 200, 75), 
  step = 25,
  max.runs = 40,
  outcome  = "~LSAP",
  conditions =  conds,
  incl.cut = 0.75,
  n.cut = 1,
  include = "?"
)

rob.calibrange(
  raw.data = ISOLATEDLIFTRAW,
  calib.data = ISOLATEDLIFTCAL,
  test.cond.raw = "C_Schneemenge_FEB",
  test.cond.calib = "LOWSNOWFEB",
  test.thresholds = c(70, 49.9, 30),
  step = 5,
  max.runs = 40,
  outcome  = "~LSAP",
  conditions =  conds,
  incl.cut = 0.75,
  n.cut = 1,
  include = "?"
)

# Testing consistency cut-off

rob.inclrange(
  data = ISOLATEDLIFTCAL,
  step = 0.01,
  max.runs = 40,
  outcome  = "~LSAP",
  conditions =  conds,
  incl.cut = 0.75,
  n.cut = 1,
  include = "?"
)

# Testing n-cut

rob.ncutrange(
  data = ISOLATEDLIFTCAL,
  step = 1,
  max.runs = 20,
  outcome  = "~LSAP",
  conditions =  conds,
  incl.cut = 0.75,
  n.cut = 1,
  include = "?"
)

# Creating test solutions

# Altering the consistency threshold

TSny1 <- minimize(data = ISOLATEDLIFTCAL,
                 outcome  = "~LSAP",
                 conditions = conds,
                 incl.cut = 0.72,
                 n.cut = 1,
                 include = "?",
                 details = TRUE, show.cases = TRUE)

TSny1

TSny2 <- minimize(data = ISOLATEDLIFTCAL2,
                 outcome  = "~LSAP",
                 conditions = conds,
                 incl.cut = 0.75,
                 n.cut = 1,
                 include = "?",
                 details = TRUE, show.cases = TRUE)


TSny2

TSny3 <- minimize(data = ISOLATEDLIFTCAL2,
                 outcome  = "~LSAP",
                 conditions = conds,
                 incl.cut = 0.75,
                 n.cut = 2,
                 include = "?",
                 details = TRUE, show.cases = TRUE)

TSny3

intersection(TSny1, TSny2, TSny3)

# Creating the test set in a list:
TSny <- list(TSny1, TSny2, TSny3)

TSny

# Obtaining Boolean expression for RC:
intersection(IS_ny, TSny1, TSny2, TSny3)

# Calculating parameters for the robust core:
rob.corefit(test_sol = TSny, 
            initial_sol = IS_ny, 
            outcome = "~LSAP")

RFny <- rob.fit(test_sol = TSny, 
                   initial_sol = IS_ny, 
                   outcome = "~LSAP") 

RFny

# Plotting the initial solution against the test set:

rob.xyplot(test_sol = TSny, 
           initial_sol = IS_ny, 
           outcome = "~LSAP", 
           fontsize = 5, 
           all_labels = FALSE,
           jitter=TRUE,
           area_lab = TRUE)

# Looking at cases against the test set:

rob.cases(test_sol = TSny, 
          initial_sol = IS_ny, 
          outcome = "~LSAP")

# Worse test solution:

rob.singletest(test_sol = TSny, initial_sol = IS_ny, outcome = "~LSAP")
